<?php

namespace app\models;

use yii\db\ActiveRecord;
use yii\web\IdentityInterface;
use Yii;

class User extends ActiveRecord implements IdentityInterface
{
    public $password; // Виртуальное поле для формы
    public $confirmPassword; // Виртуальное поле для подтверждения

    public static function tableName()
    {
        return 'user';
    }

    public function rules()
    {
        return [
            [['fullname', 'phone', 'email', 'password', 'confirmPassword'], 'required'],

            ['fullname', 'match', 'pattern' => '/^[а-яА-Я\s\-]+$/u',
                'message' => 'Только русские буквы, пробелы и дефисы'],
            ['fullname', 'string', 'max' => 100],

            ['phone', 'match', 'pattern' => '/^8\s?\(?\d{3}\)?\s?\d{3}[- ]?\d{2}[- ]?\d{2}$/',
                'message' => 'Формат: 8(999)999-99-99'],
            ['phone', 'unique', 'message' => 'Телефон уже зарегистрирован'],

            ['email', 'email'],
            ['email', 'unique', 'message' => 'Email уже используется'],

            ['password', 'string', 'min' => 8, 'message' => 'Минимум 8 символов'],

            ['confirmPassword', 'compare', 'compareAttribute' => 'password',
                'message' => 'Пароли не совпадают'],
        ];
    }

    public function attributeLabels()
    {
        return [
            'fullname' => 'ФИО',
            'phone' => 'Телефон',
            'email' => 'Email',
            'password' => 'Пароль',
            'confirmPassword' => 'Повторите пароль',
        ];
    }

    public function beforeSave($insert)
    {
        if (parent::beforeSave($insert)) {
            // Хешируем пароль перед сохранением
            if ($this->password) {
                $this->password_hash = Yii::$app->security->generatePasswordHash($this->password);
            }
            return true;
        }
        return false;
    }

    // Проверка пароля
    public function validatePassword($password)
    {
        return Yii::$app->security->validatePassword($password, $this->password_hash);
    }

    // IdentityInterface
    public static function findIdentity($id)
    {
        return static::findOne($id);
    }

    public static function findIdentityByAccessToken($token, $type = null)
    {
        return null;
    }

    public function getId()
    {
        return $this->id;
    }

    public function getAuthKey()
    {
        return null;
    }

    public function validateAuthKey($authKey)
    {
        return false;
    }

    // Поиск по телефону (для входа)
    public static function findByPhone($phone)
    {
        return static::findOne(['phone' => $phone]);
    }

    public function isAdmin()
    {
        return $this->role === 'admin';
    }
}